<?php
/**
 * Plugin Name: Comment2AIChat
 * Description: Replace WordPress default comment area (for posts only) with an AI chat using either OpenAI or DeepSeek.
 * Version: 1.0.2
 * Author: Qgenius
 * Author URI: https://www.qgenius.com/comment2aichat/
 * License: GPLv2 or later
 * Text Domain: comment2aichat
 */

// Prevent direct access
if ( ! defined('ABSPATH') ) {
    exit;
}

/**
 * Plugin activation hook
 * Create database table Comment2AIChat_Model
 */
function comment2aichat_activate() {
    global $wpdb;
    // Table name is trusted (constructed from $wpdb->prefix), so we use esc_sql() for extra safety.
    $table_name = esc_sql( $wpdb->prefix . 'Comment2AIChat_Model' );
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS `{$table_name}` (
        `ID` INT(10) NOT NULL AUTO_INCREMENT,
        `Model` VARCHAR(255) NOT NULL,
        `Apikey` VARCHAR(255) NOT NULL,
        `Prompt` TEXT NOT NULL,
        PRIMARY KEY (`ID`)
    ) $charset_collate;";

    require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );
    dbDelta($sql);

    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
    $query = "SELECT COUNT(*) FROM `{$table_name}`";
    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
    $check = $wpdb->get_var( $query );
    if ( ! $check ) {
        $wpdb->insert(
            $table_name,
            array(
                'Model'  => 'deepseek-chat',
                'Apikey' => '',
                'Prompt' => 'You are a helpful assistant.'
            )
        );
    }
}
register_activation_hook( __FILE__, 'comment2aichat_activate' );

/**
 * Plugin deactivation hook
 */
function comment2aichat_deactivate() {
    // Not removing table data on deactivate
}
register_deactivation_hook( __FILE__, 'comment2aichat_deactivate' );

/**
 * Plugin uninstall hook
 * Remove the created table
 */
function comment2aichat_uninstall() {
    global $wpdb;
    $table_name = esc_sql( $wpdb->prefix . 'Comment2AIChat_Model' );
    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
    $wpdb->query( "DROP TABLE IF EXISTS `{$table_name}`" );
}
register_uninstall_hook( __FILE__, 'comment2aichat_uninstall' );

/**
 * Disable WordPress default comment functionality for 'post' only
 */
function comment2aichat_disable_post_comments( $open, $post_id ) {
    if ( 'post' === get_post_type( $post_id ) ) {
        return false;
    }
    return $open;
}
add_filter( 'comments_open', 'comment2aichat_disable_post_comments', 10, 2 );

/**
 * Auto-append the AI Chat to the end of post content
 */
function comment2aichat_auto_append_to_post( $content ) {
    if ( is_singular('post') ) {
        $content .= do_shortcode('[comment2aichat_chat]');
    }
    return $content;
}
add_filter( 'the_content', 'comment2aichat_auto_append_to_post' );

/**
 * Add AIChat settings page under "Settings"
 */
function comment2aichat_add_settings_page() {
    add_options_page(
        'AIChat Settings',
        'AIChat',
        'manage_options',
        'comment2aichat-settings',
        'comment2aichat_settings_page_html'
    );
}
add_action( 'admin_menu', 'comment2aichat_add_settings_page' );

/**
 * Render AIChat settings page
 */
function comment2aichat_settings_page_html() {
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    global $wpdb;
    $table_name = esc_sql( $wpdb->prefix . 'Comment2AIChat_Model' );

    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
    $row = $wpdb->get_row( "SELECT * FROM `{$table_name}` WHERE ID = 1" );
    $current_model  = isset( $row->Model ) ? $row->Model : 'deepseek-chat';
    $current_api    = isset( $row->Apikey ) ? $row->Apikey : '';
    $current_prompt = isset( $row->Prompt ) ? $row->Prompt : '';

    if ( isset( $_POST['comment2aichat_submit'] ) ) {
        check_admin_referer( 'comment2aichat_update_settings' );

        // Safely handle input: check existence, unslash, then sanitize
        $user_selected_model = isset( $_POST['comment2aichat_model'] ) ? sanitize_text_field( wp_unslash( $_POST['comment2aichat_model'] ) ) : '';
        $model = ( 'o3-mini' === $user_selected_model ) ? 'o3-mini' : 'deepseek-chat';

        $apikey = isset( $_POST['comment2aichat_apikey'] ) ? sanitize_text_field( wp_unslash( $_POST['comment2aichat_apikey'] ) ) : '';
        $prompt = isset( $_POST['comment2aichat_prompt'] ) ? sanitize_textarea_field( wp_unslash( $_POST['comment2aichat_prompt'] ) ) : '';

        $wpdb->update(
            $table_name,
            array(
                'Model'  => $model,
                'Apikey' => $apikey,
                'Prompt' => $prompt,
            ),
            array( 'ID' => 1 )
        );

        echo '<div class="updated"><p>Settings updated.</p></div>';

        $current_model  = $model;
        $current_api    = $apikey;
        $current_prompt = $prompt;
    }
    ?>
    <div class="wrap">
        <h1>AIChat Settings</h1>
        <form method="POST">
            <?php wp_nonce_field( 'comment2aichat_update_settings' ); ?>
            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="comment2aichat_model">Model:</label>
                    </th>
                    <td>
                        <select id="comment2aichat_model" name="comment2aichat_model">
                            <option value="o3-mini" <?php selected( $current_model, 'o3-mini' ); ?>>OpenAI</option>
                            <option value="deepseek-chat" <?php selected( $current_model, 'deepseek-chat' ); ?>>DeepSeek</option>
                        </select>
                        <p class="description">Choose which model to use: "OpenAI" or "DeepSeek".</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="comment2aichat_apikey">API Key:</label>
                    </th>
                    <td>
                        <input type="text" id="comment2aichat_apikey" name="comment2aichat_apikey" value="<?php echo esc_attr( $current_api ); ?>" size="50"/>
                        <br>
                        <p class="description">Get an API Key from <a href="https://platform.openai.com/" target="_blank">OpenAI</a> or <a href="https://platform.deepseek.com/api_keys" target="_blank">DeepSeek</a>.</p>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="comment2aichat_prompt">Assistant’s Description (Prompt):</label>
                    </th>
                    <td>
                        <textarea id="comment2aichat_prompt" name="comment2aichat_prompt" rows="6" cols="70"><?php echo esc_textarea( $current_prompt ); ?></textarea>
                    </td>
                </tr>
            </table>
            <?php submit_button( 'Save Settings', 'primary', 'comment2aichat_submit' ); ?>
            <p style="text-align: right; font-size: smaller;">
                <a href="https://www.qgenius.com/" target="_blank">Powered by Qgenius</a>
            </p>
        </form>
    </div>
    <?php
}

/**
 * Register shortcode [comment2aichat_chat]
 */
function comment2aichat_register_shortcode() {
    ob_start(); ?>
    <div id="aichat-container-agent">
        <div id="aichat-box-agent"></div>
        <input type="text" id="aichat-user-input" class="aichat-input" placeholder="Type your question here..."/>
        <button id="aichat-submit" class="aichat-button">Send</button>
    </div>
    <?php
    return ob_get_clean();
}
add_shortcode( 'comment2aichat_chat', 'comment2aichat_register_shortcode' );

/**
 * Enqueue front-end JavaScript and CSS
 */
function comment2aichat_enqueue_scripts() {
    wp_enqueue_style(
        'comment2aichat-css',
        plugin_dir_url( __FILE__ ) . 'css/aichat.css',
        array(),
        '1.1.0',
        'all'
    );
    wp_enqueue_script(
        'comment2aichat-js',
        plugin_dir_url( __FILE__ ) . 'js/aichat.js',
        array('jquery'),
        '1.1.0',
        true
    );
    wp_localize_script( 'comment2aichat-js', 'comment2aichat_obj', array(
        'ajax_url' => admin_url( 'admin-ajax.php' ),
        'nonce'    => wp_create_nonce( 'comment2aichat_nonce' )
    ) );
}
add_action( 'wp_enqueue_scripts', 'comment2aichat_enqueue_scripts' );

/**
 * AJAX endpoint to process chat messages
 */
function comment2aichat_process_chat() {
    check_ajax_referer( 'comment2aichat_nonce', 'security' );

    $message = isset( $_POST['message'] ) ? sanitize_text_field( wp_unslash( $_POST['message'] ) ) : '';
    if ( empty( $message ) ) {
        wp_send_json_error( array('error' => 'Empty message.') );
    }

    global $wpdb;
    $table_name = esc_sql( $wpdb->prefix . 'Comment2AIChat_Model' );
    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
    $row = $wpdb->get_row( "SELECT * FROM `{$table_name}` WHERE ID = 1" );
    if ( ! $row ) {
        wp_send_json_error( array('error' => 'No model config found.') );
    }

    $api_key       = $row->Apikey;
    $system_prompt = $row->Prompt;
    $model         = $row->Model;
    
    if ( empty( $api_key ) ) {
        wp_send_json_error( array('error' => 'API key is empty. Please configure it in Settings > AIChat.') );
    }

    $messages = array(
        array("role" => "system", "content" => $system_prompt),
        array("role" => "user",   "content" => $message)
    );

    $endpoint_url = ( 'o3-mini' === $model ) ? 'https://api.openai.com/v1/chat/completions' : 'https://api.deepseek.com/chat/completions';

    $body = array(
        'model'    => $model,
        'messages' => $messages,
        'stream'   => false
    );

    $response = wp_remote_post( $endpoint_url, array(
        'headers' => array(
            'Content-Type'  => 'application/json',
            'Authorization' => 'Bearer ' . $api_key
        ),
        'body'    => wp_json_encode( $body ),
        'timeout' => 30
    ) );

    if ( is_wp_error( $response ) ) {
        wp_send_json_error( array('error' => $response->get_error_message()) );
    }

    $response_body = wp_remote_retrieve_body( $response );
    $decoded = json_decode( $response_body, true );

    if ( ! isset( $decoded['choices'][0]['message']['content'] ) ) {
        if ( isset( $decoded['error']['message'] ) ) {
            wp_send_json_error( array('error' => $decoded['error']['message']) );
        }
        wp_send_json_error( array('error' => 'Invalid response from API.') );
    }

    $assistant_reply = $decoded['choices'][0]['message']['content'];
    wp_send_json_success( array('reply' => $assistant_reply) );
}
add_action( 'wp_ajax_comment2aichat_process_chat', 'comment2aichat_process_chat' );
add_action( 'wp_ajax_nopriv_comment2aichat_process_chat', 'comment2aichat_process_chat' );
